class RequestQueue {
  constructor(concurrency = 2) {
    this.queue = [];
    this.activeRequests = 0;
    this.concurrency = concurrency;
  }

  enqueue(task) {
    return new Promise((resolve, reject) => {
      this.queue.push({ task, resolve, reject });
      this.process();
    });
  }

  async process() {
    if (this.activeRequests >= this.concurrency || this.queue.length === 0) {
      return;
    }

    this.activeRequests++;
    const { task, resolve, reject } = this.queue.shift();

    try {
      const result = await task();
      resolve(result);
    } catch (error) {
      reject(error);
    } finally {
      this.activeRequests--;
      this.process();
    }
  }
}

const downloadQueue = new RequestQueue(2); // Limit to 2 concurrent requests

// --- Context Menu Setup ---
chrome.runtime.onInstalled.addListener(() => {
  chrome.contextMenus.create({
    id: "saveToKYT",
    title: "Save to KYT",
    contexts: ["link"],
    targetUrlPatterns: ["*://*.youtube.com/watch*", "*://*.youtube.com/shorts/*"]
  });
});

chrome.contextMenus.onClicked.addListener(async (info, tab) => {
  if (info.menuItemId === "saveToKYT") {
    const videoUrl = info.linkUrl;
    
    // Get API Key from storage
    const items = await chrome.storage.sync.get({ apiKey: '' });
    const apiKey = items.apiKey;

    if (!apiKey) {
      // Notify user they need to set the key
      if (tab && tab.id) {
        chrome.tabs.sendMessage(tab.id, {
          action: 'showNotification',
          message: 'Please set your API Key in the Save to KYT extension settings first.',
          type: 'error'
        }).catch(err => console.error('Failed to send message:', err));
      }
      return;
    }

    // Enqueue download
    const downloadTask = () => handleDownload(apiKey, videoUrl);
    
    downloadQueue.enqueue(downloadTask)
      .then(() => {
        // Success feedback
        if (tab && tab.id) {
          chrome.tabs.sendMessage(tab.id, {
            action: 'showNotification',
            message: 'Video sent to KYT!',
            type: 'success'
          }).catch(err => console.error('Failed to send message:', err));
        }
      })
      .catch((error) => {
        // Error feedback
        if (tab && tab.id) {
          chrome.tabs.sendMessage(tab.id, {
            action: 'showNotification',
            message: 'KYT Error: ' + error.message,
            type: 'error'
          }).catch(err => console.error('Failed to send message:', err));
        }
      });
  }
});

chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === "downloadVideo") {
    
    // Wrap the download logic in a task function
    const downloadTask = () => handleDownload(request.apiKey, request.videoUrl);

    // Add to queue
    downloadQueue.enqueue(downloadTask)
      .then((result) => sendResponse(result))
      .catch((error) => sendResponse({ success: false, error: error.message }));
    
    return true; // Will respond asynchronously
  } else if (request.action === "checkVideo") {
    handleCheckVideo(request.apiKey, request.videoUrl)
      .then((result) => sendResponse(result))
      .catch((error) => sendResponse({ success: false, error: error.message }));
    return true;
  }
});

async function handleCheckVideo(apiKey, videoUrl) {
  try {
    const baseUrl = "https://youtube.mywebartist.eu/api/check_video";
    const params = new URLSearchParams({
      api_key: apiKey,
      url: videoUrl
    });
    
    const targetUrl = `${baseUrl}?${params.toString()}`;
    
    const response = await fetch(targetUrl, {
      method: 'GET'
    });

    if (!response.ok) {
      throw new Error(`API returned status: ${response.status}`);
    }
    
    const data = await response.json();
    return { success: true, data: data };

  } catch (error) {
    console.error("Check video failed:", error);
    return { success: false, error: error.message };
  }
}

async function handleDownload(apiKey, videoUrl) {
  try {
    const baseUrl = "https://youtube.mywebartist.eu//api/download";
    const params = new URLSearchParams({
      api_key: apiKey,
      url: videoUrl
    });
    
    const targetUrl = `${baseUrl}?${params.toString()}`;
    
    console.log("Processing API Call:", targetUrl);
    
    const response = await fetch(targetUrl, {
      method: 'GET'
    });

    if (!response.ok) {
      throw new Error(`API returned status: ${response.status}`);
    }
    
    const data = await response.json();
    
    // Check for success based on new API response structure
    if (data.status === 'queued' || data.message === 'Download queued successfully') {
      return { success: true, data: data };
    } else {
      throw new Error(data.message || 'Unknown API error');
    }

  } catch (error) {
    console.error("Download failed:", error);
    throw error; // Propagate to queue
  }
}
