const DEFAULT_API_KEY = "";

// Icons (SVG)
const ICONS = {
  download: `<svg viewBox="0 0 24 24" preserveAspectRatio="xMidYMid meet" focusable="false" style="pointer-events: none; display: block; width: 100%; height: 100%;"><g><path d="M17,18v1H6v-1H17 M6,17v1h11v-1H6z M12,3l0,11l3-3l1,1l-5,5l-5-5l1-1l3,3L10,3H12z" class="style-scope yt-icon"></path></g></svg>`,
  loading: `<svg viewBox="0 0 24 24" class="kyt-spinner"><path d="M12,4V2A10,10 0 0,0 2,12H4A8,8 0 0,1 12,4Z"/></svg>`,
  success: `<svg viewBox="0 0 24 24"><path d="M9 16.17L4.83 12l-1.42 1.41L9 19 21 7l-1.41-1.41z"/></svg>`,
  error: `<svg viewBox="0 0 24 24"><path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm1 15h-2v-2h2v2zm0-4h-2V7h2v6z"/></svg>`
};

function getApiKey() {
  return new Promise((resolve) => {
    chrome.storage.sync.get({ apiKey: DEFAULT_API_KEY }, (items) => {
      resolve(items.apiKey);
    });
  });
}

function checkVideoStatus(button, videoUrlOrGetter, isThumbnail = false) {
  getApiKey().then(apiKey => {
    if (!apiKey) return;

    let videoUrl;
    if (typeof videoUrlOrGetter === "function") {
      videoUrl = videoUrlOrGetter();
    } else {
      videoUrl = videoUrlOrGetter;
    }
    
    // Clean URL to ensure better matching (remove playlist params, timestamps, etc.)
    try {
      const urlObj = new URL(videoUrl);
      if (urlObj.hostname.includes('youtube.com') && urlObj.searchParams.has('v')) {
        // Keep only the video ID
        videoUrl = `https://www.youtube.com/watch?v=${urlObj.searchParams.get('v')}`;
      }
    } catch (e) {
      // Ignore invalid URLs
    }

    chrome.runtime.sendMessage({
      action: "checkVideo",
      apiKey: apiKey,
      videoUrl: videoUrl
    }, (response) => {
      if (chrome.runtime.lastError) {
        console.warn("KYT Check Video Error:", chrome.runtime.lastError);
        return;
      }
      
      if (response && response.success && response.data) {
        const data = response.data;
        // Check for common "saved" indicators, primarily 'exists'
        const isSaved = data.exists === true || data.saved === true || data.status === 'saved' || data.is_saved === true;
        
        if (isSaved) {
          button.classList.add("success");
          if (isThumbnail) {
            button.innerHTML = `<span class="kyt-icon">${ICONS.success}</span>`;
          } else {
            button.innerHTML = `<span class="kyt-icon">${ICONS.success}</span><span class="kyt-text">Saved!</span>`;
          }
        }
      } else {
        console.log("KYT Check Video Response:", response);
      }
    });
  });
}

function createSplitButton(videoUrlOrGetter) {
  const container = document.createElement("div");
  container.className = "kyt-split-btn-container";
  
  // Left side - Save button
  const saveBtn = document.createElement("button");
  saveBtn.className = "kyt-split-btn kyt-split-btn-save";
  saveBtn.title = "Save to KYT";
  saveBtn.innerHTML = `<span class="kyt-icon">${ICONS.download}</span><span class="kyt-text">Save</span>`;
  
  // Right side - Open website button
  const websiteBtn = document.createElement("button");
  websiteBtn.className = "kyt-split-btn kyt-split-btn-website";
  websiteBtn.title = "Open KYT Website";
  websiteBtn.innerHTML = `<span class="kyt-text">↗</span>`;
  
  checkVideoStatus(saveBtn, videoUrlOrGetter, false);
  
  // Website button click handler
  websiteBtn.addEventListener("click", (e) => {
    e.preventDefault();
    e.stopPropagation();
    window.open('https://youtube.mywebartist.eu/', '_blank');
  });
  
  // Save button click handler
  saveBtn.addEventListener("click", async (e) => {
    e.preventDefault();
    e.stopPropagation();
    
    if (saveBtn.classList.contains("loading") || saveBtn.classList.contains("success")) return;

    saveBtn.classList.add("loading");
    saveBtn.innerHTML = `<span class="kyt-icon">${ICONS.loading}</span><span class="kyt-text">Processing...</span>`;

    try {
      const apiKey = await getApiKey();
      
      let videoUrl;
      if (typeof videoUrlOrGetter === "function") {
        videoUrl = videoUrlOrGetter();
      } else {
        videoUrl = videoUrlOrGetter;
      }

      const response = await chrome.runtime.sendMessage({
        action: "downloadVideo",
        apiKey: apiKey,
        videoUrl: videoUrl
      });

      if (response && response.success) {
        saveBtn.classList.remove("loading");
        saveBtn.classList.add("success");
        saveBtn.innerHTML = `<span class="kyt-icon">${ICONS.success}</span><span class="kyt-text">Saved!</span>`;
      } else {
        throw new Error(response ? response.error : "Unknown error");
      }
    } catch (error) {
      console.error("KYT Error:", error);
      saveBtn.classList.remove("loading");
      saveBtn.classList.add("error");
      saveBtn.innerHTML = `<span class="kyt-icon">${ICONS.error}</span><span class="kyt-text">Error</span>`;
      
      setTimeout(() => {
        saveBtn.classList.remove("error");
        saveBtn.innerHTML = `<span class="kyt-icon">${ICONS.download}</span><span class="kyt-text">Save</span>`;
      }, 3000);
    }
  });
  
  container.appendChild(saveBtn);
  container.appendChild(websiteBtn);
  
  return container;
}

function createButton(videoUrlOrGetter, isThumbnail = false) {
  if (!isThumbnail) {
    // Main video button is a split button
    return createSplitButton(videoUrlOrGetter);
  }
  
  // Thumbnail buttons remain single-action
  const button = document.createElement("button");
  button.className = "kyt-download-btn-thumbnail";
  button.title = "Save to KYT";
  button.innerHTML = `<span class="kyt-icon">${ICONS.download}</span>`;
  
  checkVideoStatus(button, videoUrlOrGetter, isThumbnail);

  button.addEventListener("click", async (e) => {
    e.preventDefault();
    e.stopPropagation();
    
    if (button.classList.contains("loading") || button.classList.contains("success")) return;

    button.classList.add("loading");
    
    if (isThumbnail) {
      button.innerHTML = `<span class="kyt-icon">${ICONS.loading}</span>`;
    } else {
      button.innerHTML = `<span class="kyt-icon">${ICONS.loading}</span><span class="kyt-text">Processing...</span>`;
    }

    try {
      const apiKey = await getApiKey();
      
      let videoUrl;
      if (typeof videoUrlOrGetter === "function") {
        videoUrl = videoUrlOrGetter();
      } else {
        videoUrl = videoUrlOrGetter;
      }

      // Send message to background script
      const response = await chrome.runtime.sendMessage({
        action: "downloadVideo",
        apiKey: apiKey,
        videoUrl: videoUrl
      });

      if (response && response.success) {
        button.classList.remove("loading");
        button.classList.add("success");
        if (isThumbnail) {
          button.innerHTML = `<span class="kyt-icon">${ICONS.success}</span>`;
        } else {
          button.innerHTML = `<span class="kyt-icon">${ICONS.success}</span><span class="kyt-text">Saved!</span>`;
        }
      } else {
        throw new Error(response ? response.error : "Unknown error");
      }
    } catch (error) {
      console.error("KYT Error:", error);
      button.classList.remove("loading");
      button.classList.add("error");
      
      if (isThumbnail) {
        button.innerHTML = `<span class="kyt-icon">${ICONS.error}</span>`;
      } else {
        button.innerHTML = `<span class="kyt-icon">${ICONS.error}</span><span class="kyt-text">Error</span>`;
      }
      
      // Reset after 3 seconds
      setTimeout(() => {
        button.classList.remove("error");
        if (isThumbnail) {
          button.innerHTML = `<span class="kyt-icon">${ICONS.download}</span>`;
        } else {
          button.innerHTML = `<span class="kyt-icon">${ICONS.download}</span><span class="kyt-text">Save to KYT</span>`;
        }
      }, 3000);
    }
  });

  return button;
}

function injectMainVideoButton() {
  // Target the main video action bar - try multiple selectors for robustness
  const selectors = [
    "ytd-watch-metadata #actions #top-level-buttons-computed",
    "ytd-watch-metadata #actions",
    "ytd-video-primary-info-renderer #top-level-buttons-computed",
    "#menu-container #top-level-buttons-computed",
    "ytd-menu-renderer #top-level-buttons-computed"
  ];
  
  let actions = null;
  for (const selector of selectors) {
    actions = document.querySelector(selector);
    if (actions) {
      // Ensure it's visible
      if (actions.offsetParent !== null) {
        break;
      }
    }
  }
  
  if (!actions) return;
  
  // Check if already injected
  if (actions.querySelector(".kyt-split-btn-container")) return;

  console.log("KYT: Injecting main button into", actions);

  // For the main video, we use a getter to always fetch the current URL
  const button = createButton(() => window.location.href);
  
  // Append the split button container
  actions.appendChild(button);
}

// Fallback interval to ensure injection happens even if observer misses it
setInterval(() => {
  injectMainVideoButton();
  injectThumbnailButtons();
}, 2000);

function injectThumbnailButtons() {
  // Method 1: Hover overlay on thumbnail image (like Watch Later)
  const thumbnails = document.querySelectorAll('ytd-thumbnail:not(.ytd-video-preview)');
  
  thumbnails.forEach(thumbnailContainer => {
    // Skip if already processed for overlay
    if (thumbnailContainer.hasAttribute('data-kyt-overlay')) return;
    thumbnailContainer.setAttribute('data-kyt-overlay', 'true');
    
    const thumbnailLink = thumbnailContainer.querySelector('a#thumbnail');
    if (!thumbnailLink) return;
    
    const videoUrl = thumbnailLink.href;
    if (!videoUrl || !videoUrl.includes('youtube.com/watch')) return;
    
    // Create overlay button
    const overlayBtn = createButton(videoUrl, true);
    overlayBtn.classList.add('kyt-overlay');
    thumbnailContainer.style.position = 'relative';
    thumbnailContainer.appendChild(overlayBtn);
  });
  
  // Method 2: Button in metadata area (below thumbnail)
  const videoRenderers = document.querySelectorAll('ytd-video-renderer, ytd-grid-video-renderer, ytd-compact-video-renderer');
  
  videoRenderers.forEach(renderer => {
    // Skip if already processed for metadata
    if (renderer.hasAttribute('data-kyt-metadata')) return;
    renderer.setAttribute('data-kyt-metadata', 'true');
    
    const thumbnailLink = renderer.querySelector('a#thumbnail');
    if (!thumbnailLink) return;
    
    const videoUrl = thumbnailLink.href;
    if (!videoUrl || !videoUrl.includes('youtube.com/watch')) return;
    
    // Find metadata section
    const metadataLine = renderer.querySelector('#metadata-line, ytd-video-meta-block');
    if (metadataLine) {
      const metadataBtn = document.createElement('div');
      metadataBtn.className = 'kyt-metadata-btn';
      metadataBtn.innerHTML = '<span class="kyt-metadata-link">💾 Save to KYT</span>';
      metadataBtn.style.cursor = 'pointer';
      metadataBtn.style.marginTop = '4px';
      
      metadataBtn.addEventListener('click', async (e) => {
        e.preventDefault();
        e.stopPropagation();
        
        const apiKey = await getApiKey();
        if (!apiKey) {
          alert('Please set your API key first');
          return;
        }
        
        metadataBtn.innerHTML = '<span class="kyt-metadata-link">⏳ Saving...</span>';
        
        try {
          const response = await chrome.runtime.sendMessage({
            action: "downloadVideo",
            apiKey: apiKey,
            videoUrl: videoUrl
          });
          
          if (response && response.success) {
            metadataBtn.innerHTML = '<span class="kyt-metadata-link" style="color: #2ba640;">✓ Saved!</span>';
          } else {
            metadataBtn.innerHTML = '<span class="kyt-metadata-link" style="color: #cc0000;">✗ Error</span>';
          }
        } catch (error) {
          metadataBtn.innerHTML = '<span class="kyt-metadata-link" style="color: #cc0000;">✗ Error</span>';
        }
      });
      
      metadataLine.appendChild(metadataBtn);
    }
  });
  
  // Method 3: Icon next to three-dot menu
  const videoMenus = document.querySelectorAll('ytd-video-renderer ytd-menu-renderer, ytd-grid-video-renderer ytd-menu-renderer, ytd-compact-video-renderer ytd-menu-renderer');
  
  videoMenus.forEach(menu => {
    // Skip if already processed for menu icon
    if (menu.hasAttribute('data-kyt-menu')) return;
    menu.setAttribute('data-kyt-menu', 'true');
    
    const renderer = menu.closest('ytd-video-renderer, ytd-grid-video-renderer, ytd-compact-video-renderer');
    if (!renderer) return;
    
    const thumbnailLink = renderer.querySelector('a#thumbnail');
    if (!thumbnailLink) return;
    
    const videoUrl = thumbnailLink.href;
    if (!videoUrl || !videoUrl.includes('youtube.com/watch')) return;
    
    // Create icon button next to menu
    const menuIconBtn = createButton(videoUrl, true);
    menuIconBtn.classList.add('kyt-menu-icon');
    menuIconBtn.style.position = 'relative';
    menuIconBtn.style.marginRight = '8px';
    
    const topMetadata = menu.parentElement;
    if (topMetadata) {
      topMetadata.style.display = 'flex';
      topMetadata.style.alignItems = 'center';
      topMetadata.insertBefore(menuIconBtn, menu);
    }
  });
}

// Debounce function to limit execution frequency
function debounce(func, wait) {
  let timeout;
  return function executedFunction(...args) {
    const later = () => {
      clearTimeout(timeout);
      func(...args);
    };
    clearTimeout(timeout);
    timeout = setTimeout(later, wait);
  };
}

// Debounced injection
const debouncedInject = debounce(injectMainVideoButton, 500);
const debouncedThumbnailInject = debounce(injectThumbnailButtons, 500);

// Observer to handle dynamic content loading
const observer = new MutationObserver((mutations) => {
  let shouldInject = false;
  for (const mutation of mutations) {
    if (mutation.addedNodes.length > 0) {
      shouldInject = true;
      break;
    }
  }
  
  if (shouldInject) {
    debouncedInject();
    debouncedThumbnailInject();
  }
});

// Start observing
observer.observe(document.body, {
  childList: true,
  subtree: true
});

// Handle YouTube SPA navigation
document.addEventListener("yt-navigate-finish", () => {
  // Force injection on navigation finish
  setTimeout(injectMainVideoButton, 1000);
  setTimeout(injectThumbnailButtons, 1000);
});

// Initial injection
injectMainVideoButton();
injectThumbnailButtons();

// Listen for notification messages from background script
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'showNotification') {
    showToast(request.message, request.type);
  }
});

// Toast notification function
function showToast(message, type = 'info') {
  const toast = document.createElement('div');
  toast.textContent = message;
  
  let bgColor = '#2196F3'; // info
  if (type === 'success') bgColor = '#2ba640';
  if (type === 'error') bgColor = '#cc0000';
  
  toast.style.cssText = `
    position: fixed;
    bottom: 20px;
    right: 20px;
    background: ${bgColor};
    color: white;
    padding: 12px 20px;
    border-radius: 4px;
    z-index: 999999;
    font-family: "Roboto", sans-serif;
    font-size: 14px;
    box-shadow: 0 2px 8px rgba(0,0,0,0.3);
    animation: kyt-slide-in 0.3s ease-out;
  `;
  
  document.body.appendChild(toast);
  setTimeout(() => {
    toast.style.animation = 'kyt-slide-out 0.3s ease-in';
    setTimeout(() => toast.remove(), 300);
  }, 3000);
}
